const express = require('express');
const { authenticateToken, requireStaffOrAdmin } = require('../middleware/auth');
const { uploadProductImage, uploadProductImages, handleUploadError, deleteFile, getFileUrl } = require('../middleware/upload');
const { mysqlHelper } = require('../database/mysql-helper');
const path = require('path');
const fs = require('fs');
const { validationResult, body, param } = require('express-validator');

const router = express.Router();

// Validation rules
const idValidation = [
  param('id').isInt({ min: 1 }).withMessage('Valid product ID is required')
];

const createProductValidation = [
  body('name').trim().isLength({ min: 2, max: 100 }).withMessage('Product name must be between 2 and 100 characters'),
  body('description').optional().trim().isLength({ max: 1000 }).withMessage('Description must be less than 1000 characters'),
  body('price').isFloat({ min: 0 }).withMessage('Valid price is required'),
  body('category_id').isInt({ min: 1 }).withMessage('Valid category is required'),
  body('current_stock').optional().isInt({ min: 0 }).withMessage('Current stock must be a non-negative integer'),
  body('status').optional().isIn(['active', 'inactive']).withMessage('Status must be active or inactive')
];

const updateProductValidation = [
  body('name').optional().trim().isLength({ min: 2, max: 100 }).withMessage('Product name must be between 2 and 100 characters'),
  body('description').optional().trim().isLength({ max: 1000 }).withMessage('Description must be less than 1000 characters'),
  body('price').optional().isFloat({ min: 0 }).withMessage('Valid price is required'),
  body('category_id').optional().isInt({ min: 1 }).withMessage('Valid category is required'),
  body('current_stock').optional().isInt({ min: 0 }).withMessage('Current stock must be a non-negative integer'),
  body('status').optional().isIn(['active', 'inactive']).withMessage('Status must be active or inactive')
];

// Helper to parse images from JSON string
const parseImages = (rows, req) => {
  return rows.map(p => {
    const { images, ...product } = p;
    console.log('Raw images data for product', p.id, ':', images);
    
    let parsedImages = [];
    if (images) {
      try {
        const imageArray = JSON.parse(images);
        parsedImages = imageArray.map(img => getFileUrl(req, img, 'products'));
        console.log('Parsed images for product', p.id, ':', parsedImages);
      } catch (error) {
        console.error('Error parsing images for product', p.id, ':', error);
        parsedImages = [];
      }
    }
    
    return {
      ...product,
      images: parsedImages
    };
  });
};

// Get all products
router.get('/', authenticateToken, async (req, res) => {
  try {
    const {
      category_id, status, search, sort_by = 'created_at', sort_order = 'DESC',
      limit = 50, offset = 0
    } = req.query;

    const allowedSortFields = ['name', 'price', 'created_at', 'category_name', 'current_stock', 'status'];
    const sortField = allowedSortFields.includes(sort_by) ? sort_by : 'created_at';
    const sortDirection = sort_order.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

    let baseQuery = `FROM products p LEFT JOIN categories c ON p.category_id = c.id WHERE 1=1`;
    const params = [];
    const countParams = [];

    if (category_id) {
      baseQuery += ' AND p.category_id = ?';
      params.push(category_id);
      countParams.push(category_id);
    }
    if (status) {
      baseQuery += ' AND p.status = ?';
      params.push(status);
      countParams.push(status);
    }
    if (search) {
      baseQuery += ' AND (p.name LIKE ? OR p.description LIKE ? OR c.name LIKE ?)';
      const searchTerm = `%${search}%`;
      params.push(searchTerm, searchTerm, searchTerm);
      countParams.push(searchTerm, searchTerm, searchTerm);
    }

    let finalQuery = 'SELECT p.*, c.name as category_name ' + baseQuery;
    finalQuery += sortField === 'category_name' ? ` ORDER BY c.name ${sortDirection}, p.name ASC` : ` ORDER BY p.${sortField} ${sortDirection}`;
    finalQuery += ' LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));

    const rows = await new Promise((resolve, reject) => {
      mysqlHelper.all(finalQuery, params, (err, rows) => {
        if (err) reject(err);
        else resolve(rows);
      });
    });

    const products = parseImages(rows, req);
    const countQuery = 'SELECT COUNT(*) as total ' + baseQuery;
    
    const countRow = await new Promise((resolve, reject) => {
      mysqlHelper.get(countQuery, countParams, (err, countRow) => {
        if (err) reject(err);
        else resolve(countRow);
      });
    });


    res.json({
      products,
      total: countRow.total,
      page: Math.floor(offset / limit) + 1,
      totalPages: Math.ceil(countRow.total / limit)
    });

  } catch (error) {
    console.error('Error fetching products:', error);
    res.status(500).json({ error: 'Failed to fetch products' });
  }
});

// Get public products
router.get('/public', async (req, res) => {
  try {
    const {
      category_id, search, sort_by = 'created_at', sort_order = 'DESC',
      limit = 20, offset = 0, min_price, max_price
    } = req.query;

    const allowedSortFields = ['name', 'price', 'created_at', 'category_name'];
    const sortField = allowedSortFields.includes(sort_by) ? sort_by : 'created_at';
    const sortDirection = sort_order.toUpperCase() === 'ASC' ? 'ASC' : 'DESC';

    let baseQuery = `FROM products p LEFT JOIN categories c ON p.category_id = c.id WHERE p.status = 'active' AND p.current_stock > 0`;
    const params = [];
    const countParams = [];

    if (category_id) {
      baseQuery += ' AND p.category_id = ?';
      params.push(category_id);
      countParams.push(category_id);
    }
    if (search) {
      baseQuery += ' AND (p.name LIKE ? OR p.description LIKE ? OR c.name LIKE ?)';
      const searchTerm = `%${search}%`;
      params.push(searchTerm, searchTerm, searchTerm);
      countParams.push(searchTerm, searchTerm, searchTerm);
    }
    if (min_price) {
      baseQuery += ' AND p.price >= ?';
      params.push(parseFloat(min_price));
      countParams.push(parseFloat(min_price));
    }
    if (max_price) {
      baseQuery += ' AND p.price <= ?';
      params.push(parseFloat(max_price));
      countParams.push(parseFloat(max_price));
    }

    let finalQuery = 'SELECT p.*, c.name as category_name ' + baseQuery;
    finalQuery += sortField === 'category_name' ? ` ORDER BY c.name ${sortDirection}, p.name ASC` : ` ORDER BY p.${sortField} ${sortDirection}`;
    finalQuery += ' LIMIT ? OFFSET ?';
    params.push(parseInt(limit), parseInt(offset));

    const rows = await new Promise((resolve, reject) => {
      mysqlHelper.all(finalQuery, params, (err, rows) => {
        if (err) reject(err);
        else resolve(rows);
      });
    });

    const products = parseImages(rows, req);
    const countQuery = 'SELECT COUNT(*) as total ' + baseQuery;
    
    const countRow = await new Promise((resolve, reject) => {
      mysqlHelper.get(countQuery, countParams, (err, countRow) => {
        if (err) reject(err);
        else resolve(countRow);
      });
    });


    res.json({
      products,
      pagination: {
        total: countRow.total,
        limit: parseInt(limit),
        offset: parseInt(offset),
        has_more: countRow.total > (parseInt(offset) + parseInt(limit))
      },
      filters: { category_id, search, sort_by: sortField, sort_order: sortDirection, min_price, max_price }
    });

  } catch (error) {
    console.error('Error fetching public products:', error);
    res.status(500).json({ error: 'Failed to fetch public products.' });
  }
});

// Get product by ID
router.get('/:id', async (req, res) => {
  try {
    const productId = req.params.id;
    const query = `
      SELECT p.*, c.name as category_name
      FROM products p
      LEFT JOIN categories c ON p.category_id = c.id
      WHERE p.id = ?`;

    const row = await new Promise((resolve, reject) => {
      mysqlHelper.get(query, [productId], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });


    if (!row) return res.status(404).json({ error: 'Product not found' });
    const product = parseImages([row], req)[0];
    res.json({ product });

  } catch (error) {
    console.error('Error fetching product:', error);
    res.status(500).json({ error: 'Failed to fetch product' });
  }
});

// Create new product (admin only)
router.post('/', authenticateToken, requireStaffOrAdmin, uploadProductImages, handleUploadError, createProductValidation, async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      // Clean up uploaded files if validation fails
      if (req.files && req.files.length > 0) {
        req.files.forEach(file => deleteFile(file.path));
      }
      return res.status(400).json({ errors: errors.array() });
    }

    const { name, description, price, category_id, current_stock, status = 'active' } = req.body;

    // Check if category exists
    const category = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT id FROM categories WHERE id = ?', [category_id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!category) {
      // Clean up uploaded files if category is invalid
      if (req.files && req.files.length > 0) {
        req.files.forEach(file => deleteFile(file.path));
      }
      return res.status(400).json({ error: 'Invalid category' });
    }

    // Process uploaded images
    let imagesData = null;
    if (req.files && req.files.length > 0) {
      const imageFilenames = req.files.map(file => file.filename);
      imagesData = JSON.stringify(imageFilenames);
      console.log('Storing multiple images:', imageFilenames);
    }
    
    const result = await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'INSERT INTO products (name, description, price, category_id, current_stock, status, images, created_at, updated_at) VALUES (?, ?, ?, ?, ?, ?, ?, CURRENT_TIMESTAMP, CURRENT_TIMESTAMP)',
        [name, description, price, category_id, current_stock, status, imagesData],
        function(err, result) {
          if (err) reject(err);
          else resolve({ id: result.lastID });
        }
      );
    });

    // Get created product with category info
    const newProduct = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        `SELECT p.*, c.name as category_name 
         FROM products p 
         LEFT JOIN categories c ON p.category_id = c.id 
         WHERE p.id = ?`,
        [result.id],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });

    res.status(201).json({
      message: 'Product created successfully',
      product: {
        ...newProduct,
        images: newProduct.images ? JSON.parse(newProduct.images).map(img => getFileUrl(req, img, 'products')) : []
      }
    });

  } catch (error) {
    console.error('Error creating product:', error);
    // Clean up uploaded files on error
    if (req.files && req.files.length > 0) {
      req.files.forEach(file => deleteFile(file.path));
    }
    res.status(500).json({ error: 'Server error' });
  }
});

// Update product (admin only)
router.put('/:id', authenticateToken, requireStaffOrAdmin, uploadProductImages, handleUploadError, [...idValidation, ...updateProductValidation], async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      // Clean up uploaded files if validation fails
      if (req.files && req.files.length > 0) {
        req.files.forEach(file => deleteFile(file.path));
      }
      return res.status(400).json({ errors: errors.array() });
    }

    const { id } = req.params;
    const { name, description, price, category_id, current_stock, status, replace_images = false } = req.body;

    // Check if product exists
    const existingProduct = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT images FROM products WHERE id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!existingProduct) {
      // Clean up uploaded files if product not found
      if (req.files && req.files.length > 0) {
        req.files.forEach(file => deleteFile(file.path));
      }
      return res.status(404).json({ error: 'Product not found' });
    }

    // Check if category exists (if updating category)
    if (category_id) {
      const category = await new Promise((resolve, reject) => {
        mysqlHelper.get('SELECT id FROM categories WHERE id = ?', [category_id], (err, row) => {
          if (err) reject(err);
          else resolve(row);
        });
      });

      if (!category) {
        // Clean up uploaded files if category is invalid
        if (req.files && req.files.length > 0) {
          req.files.forEach(file => deleteFile(file.path));
        }
        return res.status(400).json({ error: 'Invalid category' });
      }
    }

    // Handle multiple images update
    let imagesToUpdate = existingProduct.images;
    if (req.files && req.files.length > 0) {
      const newImageFilenames = req.files.map(file => file.filename);
      
      if (replace_images === 'true' || replace_images === true) {
        // Replace all existing images with new ones
        if (existingProduct.images) {
          const oldImages = JSON.parse(existingProduct.images);
          for (const img of oldImages) {
            deleteFile(path.join(__dirname, '..', 'uploads', 'products', img));
          }
        }
        imagesToUpdate = JSON.stringify(newImageFilenames);
        console.log('Replacing all images with:', newImageFilenames);
      } else {
        // Add new images to existing ones
        const currentImages = existingProduct.images ? JSON.parse(existingProduct.images) : [];
        const allImages = [...currentImages, ...newImageFilenames];
        imagesToUpdate = JSON.stringify(allImages);
        console.log('Adding images to existing:', allImages);
      }
    }

    // Prepare update fields
    const updateFields = [];
    const updateValues = [];

    if (name !== undefined) {
      updateFields.push('name = ?');
      updateValues.push(name);
    }

    if (description !== undefined) {
      updateFields.push('description = ?');
      updateValues.push(description);
    }

    if (price !== undefined) {
      updateFields.push('price = ?');
      updateValues.push(price);
    }

    if (category_id !== undefined) {
      updateFields.push('category_id = ?');
      updateValues.push(category_id);
    }

    if (current_stock !== undefined) {
      updateFields.push('current_stock = ?');
      updateValues.push(current_stock);
    }

    if (status !== undefined) {
      updateFields.push('status = ?');
      updateValues.push(status);
    }

    if (imagesToUpdate !== existingProduct.images) {
      updateFields.push('images = ?');
      updateValues.push(imagesToUpdate);
    }

    // Always update the updated_at timestamp
    updateFields.push('updated_at = CURRENT_TIMESTAMP');

    if (updateFields.length === 1) { // Only updated_at
      if (req.file) {
        deleteFile(req.file.path);
      }
      return res.status(400).json({ error: 'No fields to update' });
    }

    // Build and execute update query
    const updateQuery = `UPDATE products SET ${updateFields.join(', ')} WHERE id = ?`;
    updateValues.push(id);

    await new Promise((resolve, reject) => {
      mysqlHelper.run(updateQuery, updateValues, function(err, result) {
        if (err) reject(err);
        else resolve({ changes: result.changes });
      });
    });

    // Get updated product with category info
    const updatedProduct = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        `SELECT p.*, c.name as category_name 
         FROM products p 
         LEFT JOIN categories c ON p.category_id = c.id 
         WHERE p.id = ?`,
        [id],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });


    res.json({
      message: 'Product updated successfully',
      product: {
        ...updatedProduct,
        images: updatedProduct.images ? JSON.parse(updatedProduct.images).map(img => getFileUrl(req, img, 'products')) : []
      }
    });

  } catch (error) {
    console.error('Error updating product:', error);
    // Clean up uploaded files on error
    if (req.files && req.files.length > 0) {
      req.files.forEach(file => deleteFile(file.path));
    }
    res.status(500).json({ error: 'Server error' });
  }
});

// Delete product (admin only)
router.delete('/:id', authenticateToken, requireStaffOrAdmin, idValidation, async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { id } = req.params;

    // Check if product exists and get images
    const existingProduct = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT images FROM products WHERE id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!existingProduct) {
      return res.status(404).json({ error: 'Product not found' });
    }

    // Delete product
    await new Promise((resolve, reject) => {
      mysqlHelper.run('DELETE FROM products WHERE id = ?', [id], (err, result) => {
        if (err) reject(err);
        else resolve({ changes: result.changes });
      });
    });

    // Delete image files if exists
    if (existingProduct.images) {
      try {
        const images = JSON.parse(existingProduct.images);
        for (const img of images) {
          deleteFile(path.join(__dirname, '..', 'uploads', 'products', img));
        }
      } catch (parseError) {
        console.error('Error parsing images for deletion:', parseError);
      }
    }

    res.json({ message: 'Product deleted successfully' });

  } catch (error) {
    console.error('Error deleting product:', error);
    
    // Handle foreign key constraint error
    if (error.code === 'ER_ROW_IS_REFERENCED_2') {
      return res.status(409).json({ 
        error: 'Cannot delete product because it is referenced in existing orders. Please remove it from all orders first.' 
      });
    }
    
    res.status(500).json({ error: 'Server error' });
  }
});

// Upload multiple product images
router.post('/:id/images', authenticateToken, requireStaffOrAdmin, uploadProductImages, handleUploadError, idValidation, async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      if (req.files && req.files.length > 0) {
        req.files.forEach(file => deleteFile(file.path));
      }
      return res.status(400).json({ errors: errors.array() });
    }

    if (!req.files || req.files.length === 0) {
      return res.status(400).json({ error: 'No files uploaded' });
    }

    const { id } = req.params;
    const { replace_images = false } = req.body;

    // Check if product exists and get current images
    const existingProduct = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT images FROM products WHERE id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!existingProduct) {
      // Clean up uploaded files if product not found
      req.files.forEach(file => deleteFile(file.path));
      return res.status(404).json({ error: 'Product not found' });
    }

    const newImageFilenames = req.files.map(file => file.filename);
    let finalImages;

    if (replace_images === 'true' || replace_images === true) {
      // Replace all existing images
      if (existingProduct.images) {
        const oldImages = JSON.parse(existingProduct.images);
        for (const img of oldImages) {
          deleteFile(path.join(__dirname, '..', 'uploads', 'products', img));
        }
      }
      finalImages = newImageFilenames;
    } else {
      // Add to existing images
      const currentImages = existingProduct.images ? JSON.parse(existingProduct.images) : [];
      finalImages = [...currentImages, ...newImageFilenames];
    }

    // Update product images in database
    await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'UPDATE products SET images = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [JSON.stringify(finalImages), id],
        (err) => {
          if (err) reject(err);
          else resolve();
        }
      );
    });

    // Get updated product
    const updatedProduct = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        `SELECT p.*, c.name as category_name 
         FROM products p 
         LEFT JOIN categories c ON p.category_id = c.id 
         WHERE p.id = ?`,
        [id],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });

    res.json({
      message: 'Images uploaded successfully',
      product: {
        ...updatedProduct,
        images: updatedProduct.images ? JSON.parse(updatedProduct.images).map(img => getFileUrl(req, img, 'products')) : []
      }
    });

  } catch (error) {
    console.error('Error uploading images:', error);
    if (req.files && req.files.length > 0) {
      req.files.forEach(file => deleteFile(file.path));
    }
    res.status(500).json({ error: 'Server error' });
  }
});

// Upload single product image (backward compatibility)
router.post('/:id/image', authenticateToken, requireStaffOrAdmin, uploadProductImage, handleUploadError, idValidation, async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      if (req.file) {
        deleteFile(req.file.path);
      }
      return res.status(400).json({ errors: errors.array() });
    }

    if (!req.file) {
      return res.status(400).json({ error: 'No file uploaded' });
    }

    const { id } = req.params;

    // Check if product exists and get current images
    const existingProduct = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT images FROM products WHERE id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!existingProduct) {
      deleteFile(req.file.path);
      return res.status(404).json({ error: 'Product not found' });
    }

    // Delete old images if exists
    if (existingProduct.images) {
      const oldImages = JSON.parse(existingProduct.images);
      for (const img of oldImages) {
        deleteFile(path.join(__dirname, '..', 'uploads', 'products', img));
      }
    }

    // Update product with new images
    await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'UPDATE products SET images = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [req.file ? JSON.stringify([req.file.filename]) : null, id],
        function(err, result) {
          if (err) reject(err);
          else resolve({ changes: result.changes });
        }
      );
    });


    res.json({
      message: 'Product images uploaded successfully',
      images: req.file ? [getFileUrl(req, req.file.filename, 'products')] : []
    });

  } catch (error) {
    console.error('Error uploading product images:', error);
    if (req.file) {
      deleteFile(req.file.path);
    }
    res.status(500).json({ error: 'Server error' });
  }
});

// Delete product images
router.delete('/:id/images', authenticateToken, requireStaffOrAdmin, idValidation, async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { id } = req.params;

    // Get current images
    const product = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT images FROM products WHERE id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!product) {
      return res.status(404).json({ error: 'Product not found' });
    }

    if (!product.images) {
      return res.status(404).json({ error: 'No images to delete' });
    }

    // Delete image files
    const images = JSON.parse(product.images);
    for (const img of images) {
      deleteFile(path.join(__dirname, '..', 'uploads', 'products', img));
    }

    // Update product to remove image references
    await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'UPDATE products SET images = NULL, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [id],
        function(err, result) {
          if (err) reject(err);
          else resolve({ changes: result.changes });
        }
      );
    });


    res.json({ message: 'Product images deleted successfully' });

  } catch (error) {
    console.error('Error deleting product images:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// Delete specific product images by filename
router.delete('/:id/images/:filename', authenticateToken, requireStaffOrAdmin, idValidation, async (req, res) => {
  try {
    const errors = validationResult(req);
    if (!errors.isEmpty()) {
      return res.status(400).json({ errors: errors.array() });
    }

    const { id, filename } = req.params;

    // Get current images
    const product = await new Promise((resolve, reject) => {
      mysqlHelper.get('SELECT images FROM products WHERE id = ?', [id], (err, row) => {
        if (err) reject(err);
        else resolve(row);
      });
    });

    if (!product) {
      return res.status(404).json({ error: 'Product not found' });
    }

    if (!product.images) {
      return res.status(404).json({ error: 'No images to delete' });
    }

    const images = JSON.parse(product.images);
    const imageIndex = images.findIndex(img => img === filename);

    if (imageIndex === -1) {
      return res.status(404).json({ error: 'Image not found' });
    }

    // Delete the specific image file
    deleteFile(path.join(__dirname, '..', 'uploads', 'products', filename));

    // Remove image from array
    images.splice(imageIndex, 1);

    // Update product with remaining images
    const updatedImages = images.length > 0 ? JSON.stringify(images) : null;
    await new Promise((resolve, reject) => {
      mysqlHelper.run(
        'UPDATE products SET images = ?, updated_at = CURRENT_TIMESTAMP WHERE id = ?',
        [updatedImages, id],
        function(err, result) {
          if (err) reject(err);
          else resolve({ changes: result.changes });
        }
      );
    });

    // Get updated product
    const updatedProduct = await new Promise((resolve, reject) => {
      mysqlHelper.get(
        `SELECT p.*, c.name as category_name 
         FROM products p 
         LEFT JOIN categories c ON p.category_id = c.id 
         WHERE p.id = ?`,
        [id],
        (err, row) => {
          if (err) reject(err);
          else resolve(row);
        }
      );
    });

    res.json({ 
      message: 'Image deleted successfully',
      product: {
        ...updatedProduct,
        images: updatedProduct.images ? JSON.parse(updatedProduct.images).map(img => getFileUrl(req, img, 'products')) : []
      }
    });

  } catch (error) {
    console.error('Error deleting specific image:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

// Debug endpoint to check database content
router.get('/debug/database', authenticateToken, requireStaffOrAdmin, async (req, res) => {
  try {
    
    const rows = await new Promise((resolve, reject) => {
      mysqlHelper.all('SELECT id, name, images FROM products LIMIT 10', (err, rows) => {
        if (err) reject(err);
        else resolve(rows);
      });
    });

    
    res.json({
      message: 'Database content for debugging',
      products: rows
    });

  } catch (error) {
    console.error('Error in debug endpoint:', error);
    res.status(500).json({ error: 'Server error' });
  }
});

module.exports = router;
